unit DSMain;

{ Compare Delphi source file versions for text differences,
  e.g., main.pas compared to main.~pa line by line.

  Copyright (c) 1996, by Philip Stevenson
}

interface

uses
  {$IFDEF WIN32} Windows, ComCtrls, {$ELSE} WinTypes, WinProcs, {$ENDIF}
  SysUtils, Messages, Classes, Graphics,
  Controls, Forms, Dialogs, Grids, ExtCtrls, StdCtrls, Buttons,
  ShellApi, Menus, FileCtrl, Inifiles, TabNotBk,
  DSList, FileInfo, DSComp, DSMatch;

type

  TStates = (stSetup, stReady, stBusy, stCancel, stDone); {state of program's UI}
  TINIop = (INIread, INIwrite);

  TGridForm = class(TForm)
    BotPanel: TPanel;
    DiffGrid: TStringGrid;
    OpenDialog1: TOpenDialog;
    Menu: TMainMenu;
    FileMenu: TMenuItem;
    Edit: TMenuItem;
    View: TMenuItem;
    Options: TMenuItem;
    Help: TMenuItem;
    OpenFiles: TMenuItem;
    SaveDifferenceList: TMenuItem;
    Print: TMenuItem;
    ExitApp: TMenuItem;
    Copy: TMenuItem;
    SelectAll: TMenuItem;
    File1: TMenuItem;
    File2: TMenuItem;
    ExcludeLeadingWhitespace: TMenuItem;
    AutoLoad2ndFile: TMenuItem;
    ComparetheFiles: TMenuItem;
    CaseSensitive: TMenuItem;
    SummaryReport: TMenuItem;
    Delete: TMenuItem;
    Help1: TMenuItem;
    About1: TMenuItem;
    CondensedPrint: TMenuItem;
    LabelBlankLines: TMenuItem;
    DeleteFiles: TMenuItem;
    TabbedNotebook1: TTabbedNotebook;
    OpenFilesBtn: TBitBtn;
    RunBtn: TBitBtn;
    SummaryBtn: TBitBtn;
    DiffRadioGroup: TRadioGroup;
    MatchRadioGroup: TRadioGroup;
    DownButton: TSpeedButton;
    TopButton: TSpeedButton;
    procedure FormCreate(Sender: TObject);
    procedure OpenFilesClick(Sender: TObject);
    procedure CompareClick(Sender: TObject);
    procedure SummaryBtnClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure ExitAppClick(Sender: TObject);
    procedure PrintClick(Sender: TObject);
    procedure CopytoClipBoardClick(Sender: TObject);
    procedure FileViewClick(Sender: TObject);
    procedure SaveDifferenceListClick(Sender: TObject);
    procedure DeleteClick(Sender: TObject);
    procedure SelectAllClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure DiffGridDrawCell(Sender: TObject; Col, Row: Longint;
      Rect: TRect; State: TGridDrawState);
    procedure About1Click(Sender: TObject);
    procedure DownButtonClick(Sender: TObject);
    procedure TopButtonClick(Sender: TObject);
    procedure DiffGridKeyPress(Sender: TObject; var Key: Char);
    procedure DiffGridKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure RemoveCreatedFiles;
    procedure MenuItemClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure DiffGridMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure Help1Click(Sender: TObject);
  private
    { Private declarations }
    SL1, SL2: TSourceList;  {source code list to compare}
    S1path, S2path: string; {source code paths}
    S1file, S2file: string; {source code filenames}
    MatchCount: Integer;    {# of lines that match}
    DiffCount: Integer;     {# of difference blocks}
    different: boolean;     {set if difference found }
    INIfile: string;        {INI file name}
    cancel: boolean;
    UIstate: TStates;
    BothFiles, FirstFile, SecondFile: boolean;
    AllLines, BorderLines: boolean;
    procedure DeleteRow(const RowNum: Longint);
    procedure ClearGrid;
    procedure SetButtons(Astate: TStates);
    procedure OutputDiffList(const FN: string; const printout: boolean);
    procedure GetReady;
    procedure WMDropFiles(var Msg: TWMDropFiles);
      message WM_DROPFILES;
    procedure SelectBlock;
    procedure LoadFilesAndCompare;
    procedure GetSetINIData(InOut:TINIop);
  public
    { Public declarations }
  end;

  EOpAction = class(Exception);

var
  GridForm: TGridForm;

implementation

{$R *.DFM}

{TGridForm methods}

procedure TGridForm.DeleteRow(const RowNum: Longint);
{-This code deletes a row in the grid }
var
  ir: Longint;
begin
  with DiffGrid do
  begin
    for ir := RowNum to RowCount-2 do
      Rows[ir] := Rows[ir+1];
    RowCount := RowCount-1 {delete last row}
  end
end;

procedure TGridForm.DiffGridDrawCell(Sender: TObject; Col, Row: Longint;
  Rect: TRect; State: TGridDrawState);
var
  X: integer;
  SG: TStringGrid;
  S: string;
begin
  SG := Sender as TStringGrid;
  if Row < SG.FixedRows then {always leave headers alone}
    exit;
  { Coloring of 2nd file rows }
  if (SG.Cells[2, Row] = '') and not (gdSelected in State) then
  begin
    S := SG.Cells[Col, Row];
    SG.Canvas.Brush.Color := clActiveCaption;
    SG.Canvas.Font.Color := clCaptionText;
    SG.Canvas.TextRect(Rect, Rect.Left+2, Rect.Top+2, S);
  end;
  { Centering line #'s  - (more fiddling than it's worth?)}
  if (Col = 2) or (Col = 3) then
  begin
    S := SG.Cells[Col, Row];
    X := Rect.Left +
    (Rect.Right - Rect.Left - SG.Canvas.TextWidth(S)) div 2;
    SG.Canvas.TextRect(Rect, X, Rect.Top+2, S);
  end;
end;

procedure TGridForm.ClearGrid;
begin
  with DiffGrid do
  begin
    Rows[FixedRows].Clear; { 1st available line }
    RowCount := FixedRows+1; {clear grid}
  end;
end;

procedure TGridForm.SelectBlock;
{-Select next difference block}
var
  S: string;
  SRect: TGridRect;
begin
  {find start of block}
  with DiffGRid do
  begin
    {move below non-diff area}
    Srect := Selection;
    while ('' = Cells[0, Srect.Top]) and (Srect.Bottom < RowCount) do
    begin
      inc(Srect.Top);
      inc(Srect.Bottom);
    end;
    if Srect.Bottom >= RowCount then
    begin
      MessageBeep(MB_ICONASTERISK);
      dec(Srect.Bottom);
      Selection := Srect;
      exit;
    end;
    { determine diff block}
    S := Cells[0, Srect.Top];
    while (Srect.Bottom < RowCount) and
          (S = Cells[0, Srect.Bottom]) do
      inc(Srect.Bottom);
    Srect.Left := 0;
    Srect.Right := 0; {highlight block #'s}
    dec(Srect.Bottom);
    Selection := Srect;
    if Srect.Top > 8 then
      TopRow := Srect.Top-4;
  end;
end;

procedure TGridForm.DownButtonClick(Sender: TObject);
{-Move to next block in grid }
begin
  with DiffGRid do
  begin
    if RowCount-1 = Selection.Bottom then {been here before}
    begin
      MessageBeep(MB_ICONASTERISK);
      exit
    end;
    if Row < RowCount-1 then
    begin
      Row := Row+1;
      SelectBlock;
    end;
  end;
end;

procedure TGridForm.TopButtonClick(Sender: TObject);
{-Move to first dif line in grid }
begin
  with DiffGRid do
  begin
    TopRow := FixedRows;
    Row := FixedRows;
    Col := 0;
  end;
  SelectBlock;
end;

procedure TGridForm.SetButtons(Astate: TStates);
{-Sets UI state}
const OnOff: array[stSetup..stDone, 1..3] of boolean =
    ((True,  False, False),
     (True,  True,  False),
     (False, False, False),
     (False, True, False),
     (True,   True,  True));
var
  ifDone: boolean;
begin
  UIstate := Astate;
  TabbedNotebook1.PageIndex := 0;  {must to allow focus }
  OpenFilesBtn.Enabled := OnOff[Astate, 1];
  RunBtn.Enabled := OnOff[Astate, 2];
  SummaryBtn.Enabled := OnOff[Astate, 3];
  if Astate = stCancel then
    RunBtn.Caption := '&Cancel'
  else
    RunBtn.Caption := '&Compare';

  OpenFiles.Enabled := OpenFilesBtn.Enabled;
  ComparetheFiles.Enabled := RunBtn.Enabled;
  SummaryReport.Enabled := SummaryBtn.Enabled;
  File1.Enabled := SummaryBtn.Enabled;
  File2.Enabled := SummaryBtn.Enabled;
  ifDone := SummaryBtn.Enabled and different;
  Print.Enabled := ifDone;
  SaveDifferenceList.Enabled := ifDone;
  Copy.Enabled := ifDone;
  Delete.Enabled := ifDone;
  SelectAll.Enabled := ifDone;
  DownButton.Enabled := ifDone;
  TopButton.Enabled := ifDone;
  case Astate of
    stSetup: ActiveControl := OpenFilesBtn;
    stReady: ActiveControl := RunBtn;
    stDone: ActiveControl := SummaryBtn;
  end;
  if Astate = stBusy then
    DragAcceptFiles(Handle, False)
  else
    DragAcceptFiles(Handle, True);
end;

procedure TGridForm.LoadFilesAndCompare;
{-Load source lists and build list of differences in DiffGrid from the
  two SourceLists. Then analyze what's in DiffGrid.}

type
  TEditTypes =
  (Ins, Del, Ed1, Ed2, Rep1, Rep2, Shif, Sleft, Srght, App, Same);

const
  EDTYPE: array[Ins..Same] of string[12] = (
    'Add', 'Delete', 'Edit', ' to..', 'Replace', 'with..',
    'Shift', ' left', ' right', 'Append', '');
  AddChr = 'I'; {Insert}  DelChr = 'D';
  EditChr = 'E';  Edit2Chr = 'e';
  RepChr = 'R';  Rep2Chr = 'r';
  ShiftChr = 'S';  SLeftChr = '-';  SRightChr = '+';
  AppChr = 'A';
  SameChr = '=';
  EDCHAR: array[Ins..Same] of char = (
    AddChr, DelChr, EditChr, Edit2Chr, RepChr, Rep2Chr,
    ShiftChr, SLeftChr, SRightChr, AppChr, SameChr );

  LONG_TIME = 3000; { msec. }
var
  CT: TCompareTextLines;
  msec: Longint;
  KeyPoint: Longint;

  procedure Yield(const timeto, stop: Integer; const pass: byte);
  {-Give other tasks a chance}
  var
    ticks: Longint;
  begin
    if cancel then
    begin
      cancel := False;
      raise EOpAction.Create('Process Canceled.')
    end;
    ticks := GetTickCount;
    if ticks > msec + 250 then
    begin
      msec := ticks;
      BotPanel.Caption := Format('  Pass %d  %3.1f %%',
      [pass, (timeto*100.0) / stop]);
      Application.ProcessMessages;
      if (ticks > KeyPoint) and (UIState = stBusy) then {might cancel}
      begin
        SetButtons(stCancel);
        Screen.Cursor := crDefault;
      end
    end;
  end;

  procedure BuildDifferenceList;
  {-Compare the two files line by line and put differences
   into StringGrid}
  type
    TEntryFor = (Neither, New, Old, Both);
  var
    longest, Arow: Integer;
    found: TEntryFor;
    InDiff: boolean;
    shchr: char;

    procedure CopyLine(entry: TEntryFor; const mark: char);
    {-Copies line from a SourceList to DiffGrid}
    var
      SL: TSourceList;
    begin
      if entry = Old then
        SL := SL2
      else
        SL := SL1;
      with DiffGrid do
      begin
        if Arow >= RowCount then
          RowCount := RowCount+1;
        if mark <> SameChr then
          Cells[0, Arow] := Format('%2d',[DiffCount])
        else
          Cells[0, Arow] := '';
        Cells[1, Arow] := mark;
        if entry = Both then
          Cells[4-SL.Tag, Arow] := Format('%.3d', [succ(SL2.Position)])
        else
          Cells[4-SL.Tag, Arow] := '';
        Cells[SL.Tag+1, Arow] := Format('%.3d', [succ(SL.Position)]);

        { label blank lines / load string }
        if LabelBlankLines.Checked and IsBlank(SL.StrAtPos) then
          Cells[4, Arow] := '{BLANK LINE}'
        else
          Cells[4, Arow] := SL.StrAtPos;
        inc(Arow);
      end;
      SL.NextPos;
      if entry = Both then
        SL2.NextPos;
    end;

    function LookAhead(SLA, SLB: TSourceList): boolean;
    {-Search for nearest matching string(s) in other list}
    begin
      if not SLA.BeyondRange then {search for nearest match}
      begin
        if CT.StrMatch(SLA.StrUpNext(0), SLB.StrAtPos) then
          if CT.StrMatch(SLA.StrUpNext(1), SLB.StrAtOffset(1)) then
          begin
            Result := True;
            exit;
          end;
        SLA.NextUpAhead;
      end;
      Result := False;
    end;

  procedure LoadRemainder(SL: TSourceList;
    const newOld: TEntryFor; const EdChr: char);
  begin
    if not SL.BeyondEnd then
    begin
      different := True;
      inc(DiffCount);
      while not SL.BeyondEnd do
      begin
        CopyLine(newOld, EdChr);
        Yield(SL.Position, longest, 1);
      end
    end;
  end;

  begin            {BuildDifferenceList}
    MatchCount := 0;
    different := False;
    InDiff := False;
    DiffCount := 0;
    msec := GetTickCount;
    KeyPoint := msec + LONG_TIME; { time before cancel enable }
    Arow := DiffGrid.FixedRows;
    if SL2.EndPos > SL1.EndPos then
      longest := SL2.EndPos
    else
      longest := SL1.EndPos;

    {Scan forward line by line}
    SL1.ClearPositions;
    SL2.ClearPositions;
    while not(SL1.BeyondEnd or SL2.BeyondEnd) do
    begin
      Yield(SL1.Position, longest, 1);
      CT.SetStrings(SL1.StrAtPos, SL2.StrAtPos);
      if CT.MatchAtPos then {advance to next}
      begin
        if AllLines or InDiff then {put match in grid}
        begin
          InDiff := False;
          CopyLine(Both, SameChr);
        end
        else
        begin
          SL1.NextPos;
          SL2.NextPos; {both advance}
        end;
        inc(MatchCount);
        continue;
      end;
      different := True;
      if not InDiff then
      begin
        if BorderLines and (SL1.Position > 0) and (SL2.Position > 0) and
          (DiffGrid.Cells[4, DiffGrid.RowCount-1] <>
          SL1.StrAtOffset(-1)) then {leading line}
        begin
          SL1.PrevPos;
          SL2.PrevPos;
          CopyLine(Both, SameChr);
        end;
      end;
      InDiff := BorderLines;

      { scan for nearest match pair: find next line in other list that
        matches line at Position}
      SL1.SetRange;
      SL2.SetRange;

      found := Neither;
      while (found = Neither) and not(SL1.BeyondRange and SL2.BeyondRange) do
      begin
        if LookAhead(SL1, SL2) then {look in current first}
          found := New;
        if LookAhead(SL2, SL1) then
          if found = New then
            found := Both
          else
            found := Old;
      end;
      case found of
        Neither, Both: { replace }
          begin
            if CT.NearlySame then
            begin                {see if the lines closely match}
              if CT.ExactMatch then
              begin
                if Length(SL1.StrAtPos) < Length(SL2.StrAtPos) then
                  shchr := SLeftChr
                else
                  shchr := SRightChr;
                inc(DiffCount);
                CopyLine(Old, ShiftChr);
                CopyLine(New, shchr)
              end
              else
              begin
                inc(DiffCount);
                CopyLine(Old, EditChr);
                CopyLine(New, Edit2Chr);
              end;
            end
            else
            begin
              inc(DiffCount);
              CopyLine(Old, RepChr);
              CopyLine(New, Rep2Chr);
            end;
          end;
        New:
        begin
          with DiffGrid do
          if (Cells[2, RowCount-1] = '') or
             (Cells[1, RowCount-1] <> AddChr) or
             (StrToInt(Cells[2, RowCount-1]) <> SL1.Position) then
            inc(DiffCount);
          CopyLine(New, AddChr); { + line }
        end;
        Old:
        begin
          with DiffGrid do
            if (Cells[3, RowCount-1] = '') or
              (Cells[1, RowCount-1] <> DelChr) or
              (StrToInt(Cells[3, RowCount-1]) <> SL2.Position) then
              inc(DiffCount);
          CopyLine(Old, DelChr); { - line }
        end;
      end; {case}
    end;

    { Load any remainder from longer file }
    LoadRemainder(SL2, Old, DelChr);
    LoadRemainder(SL1, New, AppChr);
  end; {BuildDifferenceList}

  procedure FormatList;
  {-  Scan grid, add diff descriptions }
  var
    index, ix2: TEditTypes;
    grow: Integer;
    mark: char;
  begin
    with DiffGrid do
    begin
      if Length(Cells[1, FixedRows]) < 1 then {a blank field is possible }
        exit;
      for grow := FixedRows to RowCount-1 do
      begin
        Yield(grow, RowCount, 2);
        mark := (Cells[1, grow])[1];

        { Label change type }
        for index := Ins to Same do
          if EDCHAR[index] = mark then
          begin
            ix2 := index;
            if FirstFile or SecondFile then
              case index of
                Ed2: ix2 := Ed1;
                Rep2: ix2 := Rep1;
                Sleft: ix2 := Shif;
                Srght: ix2 := Shif;
              end;
            Cells[1, grow] := EDTYPE[ix2];
            break;
          end;
      end
    end
  end; {FormatList}

  procedure DoRemoval;
  {-  Scan grid, remove unwanted source lines }
  var
    grow: Integer;
  begin
    with DiffGRid do
    begin
      grow := FixedRows;
      while grow <= RowCount-1 do
      begin
        Yield(grow, RowCount, 3);
        if (FirstFile and (Cells[2, grow] = '')) or
           (SecondFile and (Cells[3, grow] = '')) then
        begin
          if RowCount-1 > FixedRows then
          begin
            DeleteRow(grow);
            continue
          end
          else
            ClearGrid;
        end;
        inc(grow);
      end;
    end;
  end; {DoRemoval}

  function GetOneLiner: string;
  const
    Say: array[0..5] of string[50] =
     (('  --- (for options set)'),
      ('  There is 1 difference.'),
      ('  There are %d differences.'),
      ('  *** Files are Identical. ***'),
      ('  Lines were deleted from previous file.'),
      ('  Lines were added to current file.')
     );
  var
    S: string;
  begin
    if different then
    begin
      S := '';
      with DiffGrid do
        if (RowCount = FixedRows+1) and (Cells[0, FixedRows] = '') then
          if FirstFile then
            S := Say[4]
          else
            S := Say[5];
      if DiffCount = 1 then
        S := Say[1]+S
      else
        S := Format(Say[2], [DiffCount])+S;
    end
    else
    begin
      if ExcludeLeadingWhitespace.Checked or
        not CaseSensitive.Checked then
        S := Say[3]+Say[0]
      else
        S := Say[3]
    end;
    Result := S;
  end;

begin {LoadFilesAndCompare}
  try
    BotPanel.Caption := '';
    SL1.LoadList;
    DiffGrid.Cells[4, 0] := 'File 1, '+S1file+' compared to...';
    SL2.LoadList;
    DiffGrid.Cells[4, 1] := 'File 2, '+S2file;
    if (SL1.Count = 0) or (SL2.Count = 0) then
      ShowMessage('Empty file(s) ???')
    else
    begin
      SetButtons(stBusy);
      Screen.Cursor := crHourGlass;
      Application.ProcessMessages;
      ClearGrid;
      AllLines := MatchRadioGroup.ItemIndex = 1;
      BorderLines := MatchRadioGroup.ItemIndex = 2;
      BothFiles := DiffRadioGroup.ItemIndex = 0;
      FirstFile := DiffRadioGroup.ItemIndex = 1;
      SecondFile := DiffRadioGroup.ItemIndex = 2;

      CT := TCompareTextLines.Create(ExcludeLeadingWhitespace.Checked,
        CaseSensitive.Checked);
      try
        BuildDifferenceList;
        FormatList;
        if not BothFiles then
          DoRemoval;
      finally
        CT.Free;
      end;
      BotPanel.Caption := GetOneLiner;
      SummaryDlg.AddToSum(BotPanel.Caption);

      Screen.Cursor := crDefault;
      SetButtons(stDone);
      if DiffGrid.RowCount > DiffGrid.FixedRows + 1 then
        SelectBlock;
    end;
  except
    on E:EOpAction do
    begin
      ClearGrid;
      SetButtons(stSetup);
      BotPanel.Caption := '';
    end;
    on E:Exception do
    begin
      BotPanel.Caption := '  ERROR';
      Screen.Cursor := crDefault;
      ShowMessage(E.Message);
      SetButtons(stSetup);
      S1file := '';
      S2file := '';
      DiffGrid.Cells[4, 0] := '';
      DiffGrid.Cells[4, 1] := '';
      BotPanel.Caption := '';
    end;
  end;
end;

procedure TGridForm.CompareClick(Sender: TObject);
{-Compare the two files}
begin
  if (Sender is TBitBtn) then
  begin
    if UIstate = stCancel then
    begin
      cancel := True;
      exit;
    end
  end;
  LoadFilesAndCompare;
end;

procedure TGridForm.RemoveCreatedFiles;
var
  S: string;
begin
  if DeleteFiles.Checked then {delete old}
  begin
    S := SL1.DeleteCreatedFile;
    if S <> '' then
      BotPanel.Caption := S;
    S := SL2.DeleteCreatedFile;
    if S <> '' then
      BotPanel.Caption := S;
  end;
end;

procedure TGridForm.GetReady;
{-Determines Current/Pred; clear lists, grid}
const
  NOT_SOURCE = '.dcu.exe.res.obj.dll.bmp.ico';
var
  TempName: string;
  diftime: TDateTime;
  Fin1, Fin2: TFileInfo;

  function NotText(const S: string): boolean;
  begin
    Result := pos(LowerCase(ExtractFileExt(S)), NOT_SOURCE) <> 0;
  end;

begin
  { Check for valid file types}
  if NotText(S1file) or NotText(S2file) then
  begin
    MessageBeep(MB_ICONASTERISK);
    raise EOpAction.Create('Non-Source Files can''t be analyzed.'+#13+
    '(in this version)');
  end;
  { Swap file names to Current/Pred order}
  Fin1 := TFileInfo.Create(S1file);
  Fin2 := TFileInfo.Create(S2file);
  try
    diftime := Fin1.FileTime-Fin2.FileTime;
    if diftime < 0 then {swap}
    begin
      TempName := S1file;
      S1file := S2file;
      S2file := TempName;
    end;
    SummaryDlg.FileSum(Fin1, Fin2);
  finally
    Fin1.Free;
    Fin2.Free;
  end;
  with DiffGrid do
  begin
    ClearGrid;
    Cells[4, 0] := 'File 1: '+S1file;
    Cells[4, 1] := 'File 2: '+S2file;
    BotPanel.Caption := '  READY';
  end;
  RemoveCreatedFiles;
  SL1.SourceFile := S1file;
  SL2.SourceFile := S2file;

  { Ready everything }
  SetButtons(stReady);
end;

procedure TGridForm.WMDropFiles(var Msg: TWMDropFiles);
{-Handle drag/drop from File Manager}
var
  Num : word;
  FN : array[0..255] of Char;
  FileName: string;
begin
  with Msg do
  begin
    for Num := 0 to DragQueryFile(Drop, Cardinal(-1), nil, 0)-1 do
    begin
      DragQueryFile(Drop, Num, FN, SizeOf(FN));
      if (S1file <> '') and (S2file = '') then
      begin
        S2file := StrPas(FN);
        BotPanel.Caption := '  Received: '+S2file;
        SetFocus;
        GetReady;
      end
      else
      begin
        S1file := StrPas(FN);
        BotPanel.Caption := '  Received: '+S1file;
        S2file := '';
        if AutoLoad2ndFile.Checked then
        begin
          FileName := Get2ndFileName(S1file);
          if FileName <> '' then
          begin
            S2file := FileName;
            SetFocus;
            GetReady;
            LoadFilesAndCompare;
          end
        end
      end;
    end;
    DragFinish(Drop);
  end;
end;

procedure TGridForm.GetSetINIData(InOut:TINIop);
const
  INIstr: array[1..10, 1..2] of string[30] =
  (('Source Paths', 'Source1 Path'),
   ('Source Paths', 'Source2 Path'),
   ('Match Options', 'Case Sensitive'),
   ('Match Options', 'Exclude Leading Whitespace'),
   ('Output Options', 'Display Matches'),
   ('Output Options', 'Display Differences'),
   ('Output Options', 'Condensed Print'),
   ('Output Options', 'Label Blank Lines'),
   ('File Options', 'Auto-Load 2nd File'),
   ('File Options', 'Delete Created Files')
  );
{-Read from or write to .INI file}
begin
  with TIniFile.Create(INIfile) do
  try
    if InOut = INIread then
    begin
      S1path := ReadString(INIstr[1, 1], INIstr[1, 2], '');
      S2path := ReadString(INIstr[2, 1], INIstr[2, 2], '');
      CaseSensitive.Checked := ReadBool(INIstr[3, 1], INIstr[3, 2], False);
      ExcludeLeadingWhitespace.Checked := ReadBool(INIstr[4, 1], INIstr[4, 2], False);
      MatchRadioGroup.ItemIndex := ReadInteger(INIstr[5, 1], INIstr[5, 2], 0);
      DiffRadioGroup.ItemIndex := ReadInteger(INIstr[6, 1], INIstr[6, 2], 0);
      CondensedPrint.Checked := ReadBool(INIstr[7, 1], INIstr[7, 2], False);
      LabelBlankLines.Checked := ReadBool(INIstr[8, 1], INIstr[8, 2], False);
      AutoLoad2ndFile.Checked := ReadBool(INIstr[9, 1], INIstr[9, 2], True);
      DeleteFiles.Checked := ReadBool(INIstr[10, 1], INIstr[10, 2], False);
    end
    else
    begin
      WriteString(INIstr[1, 1], INIstr[1, 2], ExtractFilePath(S1path));
      WriteString(INIstr[2, 1], INIstr[2, 2], ExtractFilePath(S2path));
      WriteBool(INIstr[3, 1], INIstr[3, 2], CaseSensitive.Checked);
      WriteBool(INIstr[4, 1], INIstr[4, 2], ExcludeLeadingWhitespace.Checked);
      WriteInteger(INIstr[5, 1], INIstr[5, 2], MatchRadioGroup.ItemIndex);
      WriteInteger(INIstr[6, 1], INIstr[6, 2], DiffRadioGroup.ItemIndex);
      WriteBool(INIstr[7, 1], INIstr[7, 2], CondensedPrint.Checked);
      WriteBool(INIstr[8, 1], INIstr[8, 2], LabelBlankLines.Checked);
      WriteBool(INIstr[9, 1], INIstr[9, 2], AutoLoad2ndFile.Checked);
      WriteBool(INIstr[10, 1], INIstr[10, 2], DeleteFiles.Checked);
    end
  finally
    Free;
  end;
end;

procedure TGridForm.FormCreate(Sender: TObject);
const
  HEADER: array[0..4, 0..1] of string[50] =
  ( ('Blk', '#'),
    ('Change', 'Type'),
    ('File 1', 'Line #'),
    ('File 2', 'Line #'),
    ('', ' Source Statements Added, Deleted, Modified') );
var
  ic, ir: Shortint;
begin
  Height := 440;
  Width := 560;
  Caption := Application.Title;
  with DiffGrid do
  begin
    ColWidths[0] := 30;
    if Screen.Width > 640 then
      ColWidths[4] := Screen.Width-220
    else
      ColWidths[4] := 600;
    for ir := 0 to 1 do
      for ic := 0 to 4 do
        Cells[ic, ir] := HEADER[ic, ir];
    {$IFDEF WIN32}
    Options := Options + [goRowSelect];
    {$ENDIF}
  end;

  SL1 := TSourceList.Create(1);
  SL2 := TSourceList.Create(2);
  INIfile := ChangeFileExt(ExtractFilename(Application.ExeName), '.INI');
  GetSetINIData(INIread);

  { Set up default paths/file names}
  if S1path = '' then
  begin
    GetDir(0, S1path);
    S2path := S1path;
  end;
end;

procedure TGridForm.FormShow(Sender: TObject);
{-Start off point}
var
  FileName, CurDir: string;
begin
  if ParamCount > 0 then
  begin
    GetDir(0, CurDir);
    S1file := ParamStr(1);
    if ExtractFilePath(S1file) = '' then
      S1file := AddBackSlash(CurDir) + S1File;
    if ParamCount > 1 then {has priority}
      S2file := ParamStr(2)
    else if AutoLoad2ndFile.Checked then
    begin
      FileName := Get2ndFileName(S1file);
      if FileName <> '' then
        S2file := FileName
    end;
  end;
  if S2file <> '' then
  begin
    if ExtractFilePath(S2file) = '' then
      S2File := AddBackSlash(CurDir) + S2File;
    GetReady;
(*    LoadFilesAndCompare; *)
  end
  else
    SetButtons(stSetup);
end;

procedure TGridForm.FormDestroy(Sender: TObject);
begin
  SL1.Free;
  SL2.Free;
end;

procedure TGridForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  GetSetINIData(INIwrite);  {-Write to .INI file}
  RemoveCreatedFiles;
end;

procedure TGridForm.ExitAppClick(Sender: TObject);
begin
  Close
end;

procedure TGridForm.OpenFilesClick(Sender: TObject);
{-Get two file names for files to compare }
var
  TempName: string;
begin
  with OpenDialog1 do
  begin
    Title := 'Select First Source File...';
    InitialDir := S1path;
    Filter :=
      'Delphi source files|*.pas;*.dpr;*.~pa;*.~dp'+
      '|Form files|*.dfm;*.~df'+
      '|Other delphi files|*.opt;*.txt;*.dop;*.dsk;*.~op;*.~tx;*.~do;*.~ds'+
      '|Any file (*.*)|*.*';

    if not Execute then
      exit;
    TempName := FileName;
    {Now do second file}
    Title := 'Select Second Source File...';
    InitialDir := S2path;
    {$IFNDEF WIN32}   { D 2 shouldn't need this }
    if IsFormFile(TempName) then
      FilterIndex := 2
    else
      FilterIndex := 1;
    {$ENDIF}
    if AutoLoad2ndFile.Checked then
    begin
      FileName := Get2ndFileName(TempName);
      if (FileName = '') and not Execute then
        exit;
    end
    else if not Execute then
      exit;
    S2file := FileName;
    S1file := TempName;
    S1path := ExtractFilePath(S1file);
    S2path := ExtractFilePath(S2file);
  end;
  GetReady;
  LoadFilesAndCompare;
end;

procedure TGridForm.MenuItemClick(Sender: TObject);
begin
  TMenuItem(Sender).Checked := not TMenuItem(Sender).Checked;
end;

procedure TGridForm.SummaryBtnClick(Sender: TObject);
begin
  SummaryDlg.BuildSummary(SL1, SL2, different, MatchCount);
  SummaryDlg.ShowSummary;
end;

procedure TGridForm.OutputDiffList(const FN: string; const printout: boolean);
{-Output diff list to printer or file }
const
  ESC = ^[;                       { ASCII Escape }
var
  PF: TextFile;
  ir: Integer;
  Fin1, Fin2: TFileInfo;
  last: string;

  procedure SelectPrintMode(const col : Smallint);
  var
    M: string;
  begin
    case col of
       40 : M := '(s5H';     { 5 cpi }
       80 : M := '(s10H';    { 10 cpi }
      132 : M := '(s16.67H'; { 16.67 cpi }
      160 : M := '(s20H';    { 20 cpi }
    else
      exit; {.. nothing}
    end;                          { case }
    write(PF, Esc, M);
  end;

  procedure InitPrinter;
  {- PCL printer setup: select PC-8 font; perf skip on, 66 lines }
  const
    PRN_INIT_STR = Esc+'(10U'+Esc+'&l1L'+Esc+'&l66P';
  begin
    write(PF, PRN_INIT_STR);
    if CondensedPrint.Checked then
      SelectPrintMode(132)
    else
      SelectPrintMode(80); {restore it}
  end;

  procedure WriteLine(const il: Integer);
  const
    Field: array[0..4] of ShortInt = (3, 8, 7, 7, 2);
  var
    ic: Integer;
  begin
    with DiffGrid do
    begin
      for ic := 0 to ColCount-1 do
        if ic = ColCount-1 then
          writeln(PF, '  ',Cells[4, il])
        else
          write(PF, Cells[ic, il]:Field[ic]);
    end;
  end;

begin
  AssignFile(PF, FN);
  try
    Rewrite(PF);                            { open printer }
    if printout then
      InitPrinter;
    Fin1 := TFileInfo.Create(S1File);
    Fin2 := TFileInfo.Create(S2File);
    with DiffGrid do
      try
        writeln(PF, 'Source Difference List as of: ',
          FormatDateTime(DATE_TIME_FORMAT, Now));
        writeln(PF);
        writeln(PF, 'Current File:', Fin1.FileName);
        writeln(PF, Fin1.ShowFileStats);
        writeln(PF, 'Previous File:', Fin2.FileName);
        writeln(PF, Fin2.ShowFileStats);
        writeln(PF);
        { write header strings }
        for ir := 0 to FixedRows-1 do
          WriteLine(ir);
        { print diff strings }
        last := Cells[0, FixedRows];
        for ir := FixedRows to RowCount-1 do
        begin
          if not BorderLines and (Cells[0, ir] <> last) then
          begin
            last := Cells[0, ir];
            writeln(PF, '---');
          end;
          WriteLine(ir);
        end;
        if printout then
        begin
          write(PF, ^L); {form feed}
          if CondensedPrint.Checked then
            SelectPrintMode(80); {restore it}
        end;
      finally
        CloseFile(PF);                        { close printer }
        Fin1.Free;
        Fin2.Free;
      end;
  except
    on EInOutError do
      MessageDlg('Error on output of list to '+FN, mtError, [mbOk], 0);
  end;
end;

procedure TGridForm.PrintClick(Sender: TObject);
{-Prints all of the strings in the DiffGrid }
begin
(*  AssignPrn(PF);                    { assign Prn to printer } *)
  OutputDiffList('PRN', True);    { assign Prn to printer }
end;

procedure TGridForm.SaveDifferenceListClick(Sender: TObject);
begin
  with SummaryDlg.SaveDialog1 do
  begin
    FileName := ChangeFileExt(S1file, '.dif');
    if execute then
    begin
      OutputDiffList(FileName, False);
      BotPanel.Caption := Format
      ('  Difference list for %s saved to %s', [S1file,
      ExtractFileName(FileName)]);
    end
  end
end;

procedure TGridForm.CopytoClipBoardClick(Sender: TObject);
{-Copy StringGrid selection to Memo window}
var iy: integer;
begin
  with SummaryDlg do
  try
    Caption := 'Saving to Clipboard';
    SummaryMemo.Clear;
    with DiffGrid do
    begin
      SummaryMemo.Lines.Add('Lines: '+
      ' '+   Cells[2, Selection.Top]+
      '/'+   Cells[3, Selection.Top]+
      ' to '+Cells[2, Selection.Bottom]+
      '/'   +Cells[3, Selection.Bottom]);
      for iy := Selection.Top to Selection.Bottom do
        SummaryMemo.Lines.Add(DiffGrid.Cells[4, iy]);
      SummaryMemo.SelectAll;
      SummaryMemo.CopyToClipBoard;
      BotPanel.Caption := Format('  %d Lines copied',
        [Selection.Bottom-Selection.Top + 1]);
    end;
  except
    MessageDlg('Error loading Clipboard.', mtError, [mbOk], 0);
  end;
end;

procedure TGridForm.DeleteClick(Sender: TObject);
{-Copy block to clipboard then delete it}
var
  ir, iy: integer;
begin
  CopytoClipBoardClick(Sender);
  with DiffGrid do
  begin
    ir := Selection.Top;
    for iy := Selection.Top to Selection.Bottom do
      DeleteRow(ir);
  end;
end;

procedure TGridForm.SelectAllClick(Sender: TObject);
var
  SRect: TGridRect;
begin
  with DiffGrid do
  begin
    Srect.Top := FixedRows;
    Srect.Bottom := RowCount-1;
    Srect.Left := 0;
    Srect.Right := ColCount-1;
    DiffGrid.Selection := Srect;
  end
end;

procedure TGridForm.FileViewClick(Sender: TObject);
{-View source file, go to selected diff line}
var
  difline, len: Smallint;
  linstr: string[10];
begin
  with DiffGrid do
  begin
    linstr := Cells[(Sender as TMenuItem).Tag+1, Selection.Top];
    if linstr <> '' then
    begin
      difline := StrToInt(linstr);
      len := Length(Cells[4, Selection.Top])+1;
    end
    else
    begin
      difline := 0;
      len := 0;
    end;
  end;
  BotPanel.Caption := 'Loading Source file...';
  Application.ProcessMessages;
  if (Sender as TMenuItem).Tag = 1 then
    SummaryDlg.LoadMemo(SL1, S1File, difline, len)
  else
    SummaryDlg.LoadMemo(SL2, S2file, difline, len);
  BotPanel.Caption := '';
end;

procedure TGridForm.About1Click(Sender: TObject);
{-Tell user about program}
begin
  MessageDlg('Delphi Source File Difference Analyzer'+#13+
  'June, 1996'+#13,
  mtInformation, [mbOk], 0);
end;

procedure TGridForm.DiffGridKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = ^C then
    CopytoClipBoardClick(Sender);
end;

procedure TGridForm.DiffGridKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if Key = VK_INSERT then
    if ssCtrl in Shift then
      CopytoClipBoardClick(Sender);
end;

procedure TGridForm.DiffGridMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
{-Right click on selected line # to show line in source list }
var
  ACol, ARow: Longint;
  SRect: TGridRect;
begin
  if Button = mbRight then
  begin
    { if user clicked on a line number.. }
    with DiffGrid do
      if Row >= FixedRows then
      begin        { Get line number }
        MouseToCell(X, Y, ACol, ARow);
        { Select where clicked }
        Srect.Top := ARow;
        Srect.Bottom := ARow;
        Srect.Left := ACol;
        Srect.Right := ACol;
        Selection := Srect;
        if ACol = 2 then
        begin
          FileViewClick(File1);
          exit;
        end
        else if ACol = 3 then
        begin
          FileViewClick(File2);
          exit;
        end;
      end;
    MessageBeep(0)
  end
end;

procedure TGridForm.Help1Click(Sender: TObject);
begin
  SummaryDlg.ShowHelp;
end;

end.

